/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Boot Interface definition
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file  BootInterface.h
 \brief The Boot Interface is a combination of a shared memory area, and lookup tables in well known address
 spaces that facilitate communication between a user application and bootstrap applications.

 The mailbox contains the cached module startup status, as well as a custom area of ram that is available to
 implement any passing of information between user and bootstrap applications that the designer wishes.

*/

#ifndef __BOOTINTERFACE_H
#define __BOOTINTERFACE_H

/* --- INCLUDE FILES --------------------------------------------------------------------------------------*/
#include <AtomicTypes.h>    /* Atomic data type definitions                                                */
#include <ModuleDefn.h>

#define TARGET_PROCESSOR_FAMILY_MPC5XX  1
#define TARGET_PROCESSOR_FAMILY_MPC55XX 2
#ifndef __TARGET_PROCESSOR_FAMILY
    #define __TARGET_PROCESSOR_FAMILY TARGET_PROCESSOR_FAMILY_MPC5XX
#endif

#if (__TARGET_PROCESSOR_FAMILY == TARGET_PROCESSOR_FAMILY_MPC5XX)
    #include <BootInterface_MPC5xx.h>
#elif (__TARGET_PROCESSOR_FAMILY == TARGET_PROCESSOR_FAMILY_MPC55XX)
    #include <BootInterface_MPC55xx.h>
#endif

/*---- DEFINES --------------------------------------------------------------------------------------------*/
/*! Base pointers is a constant array of void pointers to constant objects */
typedef void const* const* BASEPTRARR;

/*! \brief Returns a \ref uint4 with the pointer array version number.  This version pointer is guaranteed by design to 
           always be at offset zero in the array */
#define GetPointerArrayVersion(a)           (*a[0])

/*---- TYPEDEF --------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration of the possible pointer array version */
typedef enum
{
    /*! Pointer table version zero */
    PV0,
    /*! Pointer table version one */
    PV1,
    /*! Pointer table version two */
    PV2,
    /*! Pointer table version three */
    PV3
} E_PointerVersion;

/*! \brief Enumeration of types of runtime queries for Woodward hard boot code */
typedef enum
{
    /*! pRuntimeData points to a S_BootVersion structure */
    MRTQ_BOOTVERSION,
    /*! pRuntimeData points to a S_MotoTronBootObjectRequest structure */
    MRTQ_BOOTOBJECTREQUEST,
    /*! pRuntimeData points to a byte RegionID on entry and S_CDRIF_FLASHLogistics structure om exit */
    /*  Null pointer if error */
    MRTQ_BOOT_CDRIF_LOGISTICS_REQUEST,
    /*! pRuntimeData points to S_CDRIF_ParameterPassPassword which is then filled with the Password Calc */
    /*  Null pointer if error */
    MRTQ_BOOT_CDRIF_PASSWORD_REQUEST,
    /*! pRuntimeData points to uint4 Version number (Maj*100 + Min) */
    /*  Null pointer if error */
    MRTQ_BOOT_CDRIF_VERSION_REQUEST,
    /*! pRuntimeData points to uint4 Version number (Maj*100 + Min) */
    /*  Null pointer if error */
	MRTQ_BOOT_KWD_DAF_VERSION_REQUEST,

    NUM_MRTQ_TYPES
} E_MotoTronRuntimeQuery;

/*! \brief The type of an entry address pointer */
typedef void (* PFNENTRYPOINT)(void);

/*! \brief The type of a MotoTron Runtime Query function */
typedef bool1 (* PFNMOTOTRONHARDBOOTQUERY)(E_MotoTronRuntimeQuery query, void* pRuntimeData);

/*! \brief The Hardware BootStrap Version structure. Use with MRTQ_BOOTVERSION query */
typedef struct
{
    uint4  u4BootVersionMajor;        /*!< The Hardware BootStrap Major Version */
    uint4  u4BootVersionMinor;        /*!< The Hardware BootStrap Minor Version */
} S_BootVersion;

/*! \brief The boot object request structure. Use with MRTQ_BOOTOBJECTREQUEST query */
typedef struct
{
    uint4  in_u4RequestedSize;        /*!< Set on entry to the requested size of the object buffer */
    void*  out_pRequestBuffer;        /*!< Returns a pointer to a buffer of at least the requested size  */
} S_MotoTronBootObjectRequest;

/*==== ModuleConfiguration Pointers =======================================================================*/
#define MODULECONFIGURATION_POINTERS

/*! \brief Enumeration of the Version 0 Module Config Pointers.  See \ref ModulePointersArr for an example. */
typedef enum
{
    /*! Pointer to StructVersion - \ref uint4 */
    MC_PV0_PU4_VERSION = 0,
    /*! Pointer to module config execution entry address - \ref PFNENTRYPOINT */
    MC_PV0_PFN_ENTRY,
    /*! Pointer to module configuration definition - struct \ref S_ModuleConfigDefn */
    MC_PV0_S_CONFIGURATIONDEFN,
    /*! Pointer to crc lookup table - \ref uint4* */
    MC_PV0_UI4ARR_CRCLOOKUPTABLE,
    NUM_PV0_MODULECONFIG_POINTERS
} E_V0_ModuleConfigPointers;

/*! \brief Structure Version 0 Module Config Pointers. An alternative way to lookup the contents of a \ref BASEPTRARR array. */
typedef struct
{
    /*! Pointer to StructVersion (equivalent to looking up the \ref BASEPTRARR array at index MC_PV0_PU4_VERSION) */
    uint4 const*                pu4Version;
    /*! Pointer to module config execution entry address (equivalent to looking up the \ref BASEPTRARR array at index MC_PV0_PFN_ENTRY) */
    PFNENTRYPOINT               pfnEntry;
    /*! Pointer to module configuration definition (equivalent to looking up the \ref BASEPTRARR array at index MC_PV0_S_CONFIGURATIONDEFN) */
    S_ModuleConfigDefn const*   pModuleDefn;
    /*! Pointer to crc lookup table (equivalent to looking up the \ref BASEPTRARR array at index MC_PV0_UI4ARR_CRCLOOKUPTABLE) */
    uint4 const* const*         pCRCTableArr;
} S_V0_ModuleConfigPointers;

/*! \brief Enumeration of the Version 1 Module Config Pointers.  See \ref ModulePointersArr for an example. */
typedef enum E_V1_ModuleConfigPointers
{
    /*! Pointer to StructVersion - \ref uint4 */
    MC_PV1_PU4_VERSION = 0,
    /*! Pointer to module config execution entry address - \ref PFNENTRYPOINT */
    MC_PV1_PFN_ENTRY,
    /*! Pointer to module configuration definition - struct \ref S_ModuleConfigDefn */
    MC_PV1_S_CONFIGURATIONDEFN,
    /*! Pointer to crc lookup table - \ref uint4* */
    MC_PV1_UI4ARR_CRCLOOKUPTABLE,
    /*! Pointer to CRC32 initialize function. */
    MC_PV1_CRC32_INITIALIZE,
    /*! Pointer to CRC32 accumulate function */
    MC_PV1_CRC32_ACCUMULATE,
    /*! Pointer to CRC32 accumulate using a virtual memory read function */
    MC_PV1_CRC32_ACCUMULATE_READ,
    /*! Pointer to the CRC32 return value function */
    MC_PV1_CRC32_RETURNVALUE,
    /*! Pointer to the CRC32 compute function. This function will calculate the entire CRC32 value where as
        the INITIALIZE, ACCUMULATE and RETURNVALUE functions do not */
    MC_PV1_CRC32_COMPUTE,
    /*! Pointer to the CRC32 compute function. This function will calculate the entire CRC32 value using a virtual
        read to access the memory */
    MC_PV1_CRC32_COMPUTE_READ,

    NUM_PV1_MODULECONFIG_POINTERS
} E_V1_ModuleConfigPointers;

typedef void (*PFN_CRC32_INITIALIZE)(uint4* const out_pCRC);
typedef void (*PFN_CRC32_ACCUMULATE)(uint4* const pCRCValue, void const* Address, uint4 uNumBytes);
typedef void (*PFN_CRC32_ACCUMULATEWITHVIRTUALREAD)(uint4* const pCRCValue,
                                                    void const* Address,
                                                    uint4 uNumBytes,
                                                    uint1 (*pfnGetByte)(uint1 const* const pByte));
typedef uint4 (*PFN_CRC32_RETURNVALUE)(uint4 const* const pCRC);
typedef uint4 (*PFN_CRC32_COMPUTE)(uint1 const* pAddr, uint4 uNumBytes);
typedef uint4 (*PFN_CRC32_COMPUTEWITHVIRTUALREAD)(uint1 const* pAddr,
                                                 uint4 uNumBytes,
                                                 uint1 (*pfnGetByte)(uint1 const* const pByte));

/*! \brief Structure Version 1 Module Config Pointers. An alternative way to lookup the contents of a \ref BASEPTRARR array. */
typedef struct S_V1_ModuleConfigPointers
{
    /*! Pointer to StructVersion (equivalent to looking up the \ref BASEPTRARR array at index MC_PV0_PU4_VERSION) */
    uint4 const*                pu4Version;
    /*! Pointer to module config execution entry address (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_PFN_ENTRY) */
    PFNENTRYPOINT               pfnEntry;
    /*! Pointer to module configuration definition (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_S_CONFIGURATIONDEFN) */
    S_ModuleConfigDefn const*   pModuleDefn;
    /*! Pointer to crc lookup table (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_UI4ARR_CRCLOOKUPTABLE) */
    uint4 const* const*         pCRCTableArr;
    /*! Pointer to crc init function (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_CRC32_INITIALIZE) */
    PFN_CRC32_INITIALIZE        pfnCRC32Initialize;
    /*! Pointer to crc accumulate function (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_CRC32_ACCUMULATE) */
    PFN_CRC32_ACCUMULATE        pfnCRC32Accumulate;
    /*! Pointer to crc accumulate function (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_CRC32_ACCUMULATE_READ) */
    PFN_CRC32_ACCUMULATEWITHVIRTUALREAD pfnCRC32AccumulateWithVirtualRead;
    /*! Pointer to crc return value function (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_CRC32_RETURNVALUE) */
    PFN_CRC32_RETURNVALUE       pfnCRC32ReturnValue;
    /*! Pointer to crc computation function (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_CRC32_COMPUTE) */
    PFN_CRC32_COMPUTE           pfnCRC32Compute;
    /*! Pointer to crc computation function (equivalent to looking up the \ref BASEPTRARR array at index MC_PV1_CRC32_COMPUTE_READ) */
    PFN_CRC32_COMPUTEWITHVIRTUALREAD pfnCRC32ComputeWithVirtualRead;
} S_V1_ModuleConfigPointers;

/*==== Hardware Bootstrap Pointers ========================================================================*/
#define HARDWARE_BOOTSTRAP_POINTERS

/*! \brief Enumeration of the Version 0 Hardware BootStrap Pointers */
typedef enum
{
    /*! Pointer to StructVersion - \ref uint4 */
    HWB_PV0_PU4_VERSION = 0,
    /*! Pointer to Woodward hardware bootstrap entry address - \ref PFNENTRYPOINT */
    HWB_PV0_PFN_MOTOTRONENTRY,
    /*! Pointer to ServLink entry address - \ref PFNENTRYPOINT */
    HWB_PV0_PFN_SERVLINKENTRY,
    /*! Pointer to a MotoTron Runtime Query function - \ref PFNMOTOTRONHARDBOOTQUERY */
    HWB_PV0_PFN_MOTOTRONRUNTIMEQUERY,
    NUM_PV0_HARDWAREBOOT_POINTERS
} E_V0_HardwareBootPointers;

/*! \brief Structure Version 0 Hardware BootStrap Pointers. An alternative way to lookup the contents of a \ref BASEPTRARR array. */
typedef struct
{
    /*! Pointer to StructVersion (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV0_PU4_VERSION) */
    uint4 const*                pu4Version;
    /*! Pointer to MotoTron Bootstrap execution entry address (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV0_PFN_MOTOTRONENTRY) */
    PFNENTRYPOINT               pfnMotoTronBootEntry;
    /*! Pointer to ServLink execution entry address (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV0_PFN_SERVLINKENTRY) */
    PFNENTRYPOINT               pfnServLinkBootEntry;
    /*! Pointer to MotoTron runtime query function (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV0_PFN_MOTOTRONRUNTIMEQUERY) */
    PFNMOTOTRONHARDBOOTQUERY    pfnMotoTronHardBootQuery;
} S_V0_HardwareBootPointers;

/*! \brief Enumeration of the Version 1 Hardware BootStrap Pointers */
typedef enum
{
    /*! Pointer to StructVersion - \ref uint4 */
    HWB_PV1_PU4_VERSION = 0,
    /*! Pointer to Woodward hardware bootstrap entry address - \ref PFNENTRYPOINT */
    HWB_PV1_PFN_MOTOTRONENTRY,
    /*! Pointer to a MotoTron Runtime Query function - \ref PFNMOTOTRONHARDBOOTQUERY */
    HWB_PV1_PFN_MOTOTRONRUNTIMEQUERY,
    NUM_PV1_HARDWAREBOOT_POINTERS
} E_V1_HardwareBootPointers;

/*! \brief Structure Version 1 Hardware BootStrap Pointers. An alternative way to lookup the contents of a \ref BASEPTRARR array. */
typedef struct
{
    /*! Pointer to StructVersion (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV1_PU4_VERSION) */
    uint4 const*                pu4Version;
    /*! Pointer to MotoTron Bootstrap execution entry address (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV1_PFN_MOTOTRONENTRY) */
    PFNENTRYPOINT               pfnMotoTronBootEntry;
    /*! Pointer to MotoTron runtime query function (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV1_PFN_MOTOTRONRUNTIMEQUERY) */
    PFNMOTOTRONHARDBOOTQUERY    pfnMotoTronHardBootQuery;
} S_V1_HardwareBootPointers;

/*! \brief Enumeration of the Version 2 Hardware BootStrap Pointers */
typedef enum
{
    /*! Pointer to StructVersion - \ref uint4 */
    HWB_PV2_PU4_VERSION = 0,
    /*! Pointer to MotoTron hardware bootstrap entry address - \ref PFNENTRYPOINT */
    HWB_PV2_PFN_MOTOTRONENTRY,
    /*! Pointer to a MotoTron Runtime Query function - \ref PFNMOTOTRONHARDBOOTQUERY */
    HWB_PV2_PFN_MOTOTRONRUNTIMEQUERY,
    /*! Pointer to a MotoTron File system image write function - ref */
    HWB_PV2_PFN_FILESYSTEM_WRITEIMAGE,
    /*! Pointer to a MotoTron File system image verify function */
    HWB_PV2_PFN_FILESYSTEM_VERIFYIMAGE,
    /*! Pointer to a MotoTron File system image read function */
    HWB_PV2_PFN_FILESYSTEM_READIMAGE,
    /*! Pointer to a MotoTron File system image execute function */
    HWB_PV2_PFN_FILESYSTEM_EXECUTEIMAGE,
    /*! Pointer to a MotoTron File system image erasure/delete function */
    HWB_PV2_PFN_FILESYSTEM_ERASEIMAGE,

    NUM_PV2_HARDWAREBOOT_POINTERS
} E_V2_HardwareBootPointers;

struct S_NANDFlashDevice;
typedef sint4 (*PFNMOTOTRONFILESYSWRITE)(struct S_NANDFlashDevice const*,
                                         uint1 const* in_pImage,
                                         sint4 nImageByteSize,
                                         uint4 nImageId,
                                         NativeBool bIsExecutable,
                                         void* pStartAddr,
                                         uint1* pScratchPad,
                                         uint4* inout_pBytesInScratchPad);

typedef sint4 (*PFNMOTOTRONFILESYSVERIFY)(struct S_NANDFlashDevice const*,
                                          uint1 const* in_pImage,
                                          sint4 nImageByteSize,
                                          uint4 nImageId,
                                          uint1* pScratchPad,
                                          uint4* inout_pBytesInScratchPad);

typedef sint4 (*PFNMOTOTRONFILESYSREAD)(struct S_NANDFlashDevice const*,
                                        uint4 nImageId,
                                        uint4 nByteOffset,
                                        uint1* out_pData,
                                        uint4 uBytesToRead,
                                        uint4* out_BytesRead);

typedef sint4 (*PFNMOTOTRONFILESYSEXECUTE)(struct S_NANDFlashDevice const*,
                                           uint4 nImageId,
                                           uint1* pScratchPad,
                                           uint4* inout_pBytesInScratchPad,
                                           void (*pfOnExecuteImminent)(void));

typedef sint4 (*PFNMOTOTRONFILESYSERASE)(struct S_NANDFlashDevice const*, uint4 nImageId);

/*! \brief The called function requires the flash driver to exist */
#define ERROR_BOOT_NO_FLASH_DRV             -1
/*! \brief An internal buffer was insufficient to hold the required information. This is not an expected error
           return and suggests an internal problem */
#define ERROR_BOOT_BUFFER_OVERRUN           -2
/*! \brief The NAND Flash Hardware did not pass its start-up checks */
#define ERROR_BOOT_NAND_HW_BAD              -3
/*! \brief The boot code does not support the NAND flash hardware that has been populated on the module */
#define ERROR_BOOT_UNSUPPORTED_NAND_DEVICE  -4
/*! \brief A page of a NAND flash failed to program. The block containing that page would have been marked bad */
#define ERROR_BOOT_NAND_WR_FAIL             -5
/*! \brief The image could not be written. There were insufficient free good blocks available to hold the image */
#define ERROR_BOOT_IMAGE_WRITE              -6
/*! \brief The supplied scratch pad was too small for the NAND Flash device driver to operate with */
#define ERROR_BOOT_SCRATCH_PAD_SIZE         -7
/*! \brief The supplied image failed to verify with the image on the flash */
#define ERROR_BOOT_IMAGE_VERIFY             -8
/*! \brief The requested image could not be found */
#define ERROR_BOOT_IMAGE_NOT_FOUND          -9
/*! \brief The image id supplied with the call was not valid for that call. For example, attempting to erase
           image 0, which is the soft boot image, is prohibited. It can only be over written */
#define ERROR_BOOT_BAD_IMAGE_ID             -10

/*! \brief Structure Version 2 Hardware BootStrap Pointers. An alternative way to lookup the contents of a \ref BASEPTRARR array. */
typedef struct
{
    /*! Pointer to StructVersion (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PU4_VERSION) */
    uint4 const*                pu4Version;
    /*! Pointer to MotoTron Bootstrap execution entry address (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PFN_MOTOTRONENTRY) */
    PFNENTRYPOINT               pfnMotoTronBootEntry;
    /*! Pointer to MotoTron runtime query function (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PFN_MOTOTRONRUNTIMEQUERY) */
    PFNMOTOTRONHARDBOOTQUERY    pfnMotoTronHardBootQuery;
    /*! Pointer to MotoTron Boot File Image Write (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PFN_FILESYSTEM_WRITEIMAGE) */
    PFNMOTOTRONFILESYSWRITE     pfnMotoTronFileSysWrite;
    /*! Pointer to MotoTron Boot File Image Write (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PFN_FILESYSTEM_VERIFYIMAGE) */
    PFNMOTOTRONFILESYSVERIFY    pfnMotoTronFileSysVerify;
    /*! Pointer to MotoTron Boot File Image Write (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PFN_FILESYSTEM_READIMAGE) */
    PFNMOTOTRONFILESYSREAD      pfnMotoTronFileSysRead;
    /*! Pointer to MotoTron Boot File Image Write (equivalent to looking up the \ref BASEPTRARR array at index HWB_PV2_PFN_FILESYSTEM_EXECUTEIMAGE) */
    PFNMOTOTRONFILESYSEXECUTE   pfnMotoTronFileSysExecute;
} S_V2_HardwareBootPointers;


/*! \brief Enumeration of the startup status */
/* Name: E_BootStatus ClassID:ENUMDEC EnumDflt:"BootStatus Unknown" */
typedef enum
{
    /*! Power on reset */
    BOOTSTATUS_COLD,           /* EnumTxt:"Cold Start" */
    /*! Reset other than a Power On Reset */
    BOOTSTATUS_WARM            /* EnumTxt:"Warm Start" */
} E_BootStatus;

/*! \brief The BootMailBox structure implementation */
typedef struct
{
    uint1  u1BootStatus;              /*!< A value of type \ref E_BootStatus */
    uint1  pBootCustom[15];           /*!< An array that may hold any type of custom information */
} S_BootMailBox;

/*! \brief Enumeration of the Version 0 Dynamically Loadable Application Config Pointers. */
typedef enum
{
    /*! Pointer to StructVersion - \ref uint4 */
    DLA_PV0_PU4_VERSION = 0,
    /*! Pointer to application execution entry address - \ref PFNENTRYPOINT */
    DLA_PV0_PFN_ENTRY,
    /*! Pointer to the application version string. */
    DLA_PV0_STR_APPLICATIONVERSION,
    NUM_PV0_DYNAMICLOADABLEAPPLICATION_POINTERS
} E_V0_DynamicallyLoadableApplicationPointers;

#pragma pack()

/*---- FUNCTION PROTOTYPES --------------------------------------------------------------------------------*/

/* Expose HBoot Query Function for External Libraries */
PFNMOTOTRONHARDBOOTQUERY GetRuntimeQueryAddress(void);

#endif /* __BOOTINTERFACE_H */


/*---- END OF FILE ----------------------------------------------------------------------------------------*/

